{*******************************************************}
{                                                       }
{                 BT Library 0.01                       }
{                 Utilities                             }
{                                                       }
{         : TBTFileFinder                               }
{         Copyright (c) 1999 Theodoros Bebekis          }
{                                                       }
{*******************************************************}

{-------------------------------------------------------
 STATE: Freeware
        No restrictions,
        No guaranties
        Use it at your own risk!
--------------------------------------------------------}

{----------------------------------------------------------------------------------------
                                     HISTORY
-----------------------------------------------------------------------------------------
 Version   Date          Changes - Additions                                   Added by
-----------------------------------------------------------------------------------------
 0.01      08.01.1999    Initial Version

-----------------------------------------------------------------------------------------}


{
 Description : The TBTFileFinder class is a direct TOBject descendant and it is
               a file finder. Using this class its easy to build a dialog like
               the Windows FindFiles dialog but its actual virtue is that it returns
               the seaching results as a string list
 Target      : Delphi 2-3-4;
 State       : Freeware
 Properties  :
              IsSearching          : boolean, True when a searching is in progress
              FoldersList          : TStringList, a list of folders to search
              ChosenFilesList      : TStringList, a list of filenames to look for
              ChosenExtList        : TStringList, a list of extensions to look for
              IndividualFilesList  : TStringList, a full path filenames to look for
              ResultFileList       : TStringList, the result list having the below format:
                                     F or D to indicate file or directory found,
                                     the full path,
                                     the size,
                                     the creation datetime,
                                     the last access datetime,
                                     the last write datetime,
                                     and commas between them,
                                     The helper routine ParseFileInfoString can be use to parse this
                                     string to individual parts
              TotalFoundSize       : Cardinal, the total found size in bytes
              SearchMode           : TSearchMode enumerated type, can have one of the below values:
                                     smAll      : searches all folders in folder list and their subs
                                                  for any file, like *.*
                                     smSelected : searches only the exact paths found in the IndividualFilesList
                                     smCriteria : searches all folders in folder list based on the criteria
                                                  defined by the SearchCriteria set property
              SearchCriteria       : TSearchCriteria set type, can have none or more of the below values:
                                     scFileName,
                                     scExtension,
                                     scSubFoldersToo,
                                     scNewerThan,
                                     scOlderThan,
                                     scLargerThan,
                                     scSmallerThan
              NewerThan            : integer
              OlderThan            : integer
              LargerThan           : Cardinal
              SmallerThan          : Cardinal
                                     read the comments in the Get_FilesFromFolder routine below on how
                                     to set these properties
              SearchedFiles        : Cardinal
              SearchedFolders      : Cardinal


methods      : function Execute : boolean;      fires up a searching
               procedure Stop;                  stops a searching
               procedure ClearLists;            clears all list after a searching, do not forget to make
                                                this call between two calls to the Execute function

Author       : Theodoros Bebekis
               Greece
               e-mail : bebekis@otenet.gr
    
}



unit BTFF;




{=============================== INTERFACE ======================================}
interface

uses
  SysUtils,
  Classes;

{============================= DECLARATION =====================================}

const

 KBYTE  = Sizeof(Byte) shl 10;
 MBYTE  = KBYTE        shl 10;

 FILE_NOT_EXISTS_STR                = 'Unable to find the file : ' + #13#10 + #13#10  +   '* %s *'
                                        + #13#10  +  #13#10  + 'Process aborted.';
 FOLDER_NOT_EXISTS_STR              = 'Unable to find the folder : ' + #13#10 + #13#10  +   '* %s *'
                                        + #13#10  +  #13#10  + 'Process aborted.';
 OLDER_GREATER_THAN_NEWER_STR      = 'The defined older datetime stamp is greater than the newer defined datetime stamp.' + #13#10  +
                                     'When both older and newer datetime stamp criteria are defined' + #13#10  +
                                     'the older can not be greater than the newer.'+ #13#10  +
                                     'Condition : older <= filedate <= newer';
 LARGER_GREATER_THAN_SMALLER_STR   = 'The defined larger size is greater than the smaller defined size.' + #13#10  +
                                     'When both smaller and larger size criteria are defined' + #13#10  +
                                     'the larger can not be greater than the smaller.'+ #13#10  +
                                     'Condition : larger <= filesize <= smaller';

type

  TSearchMode = (smAll, smSelected, smCriteria);
  TSearchCriterion =  (scFileName, scExtension, scSubFoldersToo, scNewerThan, scOlderThan, scLargerThan, scSmallerThan);
  TSearchCriteria = set of TSearchCriterion;


  TAbortMethod = function : boolean of object;

  
  EFFError = class(Exception);




{===============================================================================
///                             TBTFileFinder                                 ///
===============================================================================}
  TBTFileFinder = class
  private
    FIsSearching              : boolean;

    FIndividualFilesList    : TStringList;        { individual files to find }
    FFoldersList            : TStringList;        { folders to choose files from }
    FChosenFilesList        : TStringList;        { file names to use as selection criteria }
    FChosenExtList          : TStringList;        { extensions to use as selection criteria }
    FResultFileList         : TStringList;        { holds the full path for any file meet the criteria }
    
    FAbort                  : boolean;

    FTotalFoundSize         : Cardinal;
    FSearchMode             : TSearchMode;        { smAll, smSelected, smCriteria }
    FSearchCriteria         : TSearchCriteria;    { set of TReadCriterion = (scFileName, scExtension etc.)}

    FNewerThan              : integer;
    FOlderThan              : integer;
    FLargerThan             : Cardinal;
    FSmallerThan            : Cardinal;

    FSearchedFiles          : Cardinal;
    FSearchedFolders        : Cardinal;


    { helper routines }
    procedure Find_All;
    procedure Find_Selected;
    procedure Find_Criteria;


    function  StopSearching:boolean;
  public
    property IsSearching          : boolean read FIsSearching;

    property IndividualFilesList  : TStringList read FIndividualFilesList  write FIndividualFilesList ;
    property FoldersList          : TStringList read FFoldersList  write FFoldersList ;
    property ChosenFilesList      : TStringList read FChosenFilesList  write FChosenFilesList ;
    property ChosenExtList        : TStringList read FChosenExtList  write FChosenExtList ;
    property ResultFileList       : TStringList read FResultFileList;

    property TotalFoundSize       : Cardinal read FTotalFoundSize;
    property SearchMode           : TSearchMode read FSearchMode  write FSearchMode ;
    property SearchCriteria       : TSearchCriteria read FSearchCriteria  write FSearchCriteria ;

    property NewerThan            : integer read FNewerThan  write FNewerThan ;
    property OlderThan            : integer read FOlderThan  write FOlderThan ;
    property LargerThan           : Cardinal read FLargerThan  write FLargerThan ;
    property SmallerThan          : Cardinal read FSmallerThan  write FSmallerThan ;

    property SearchedFiles        : Cardinal read FSearchedFiles;
    property SearchedFolders      : Cardinal read FSearchedFolders;


    function Execute : boolean;
    procedure Stop;
    procedure ClearLists;
    constructor Create;
    destructor Destroy; override;
  end;





procedure ParseFileInfoString(S: string; var Name, Folder, Size, FileType, CreationTime, LastAccessTime, LastWriteTime: string);



{============================ IMPLEMENTATION ====================================}

implementation

uses
  Windows,
  Forms,
  Dialogs,
  FileCtrl;



{---------------------------------------------------------------------------------}
procedure RaiseFFError(const Msg: string);
begin
  raise EFFError.Create(Msg);
end;
{---------------------------------------------------------------------------------}
procedure RaiseFFErrorFmt(const Msg: string; const Args: array of const);
begin
  raise EFFError.CreateFmt(Msg, Args);
end;


{----------------------------------------------------------------------------------
 Description    : this routine can be used to parse the FileList contents after
                  a successful call to the Get_FilesFromFolder function.
                  Pass any FileList.Item as the S parameter;
 Error checking : NO
-----------------------------------------------------------------------------------}
procedure ParseFileInfoString(S: string; var Name, Folder, Size, FileType, CreationTime, LastAccessTime, LastWriteTime: string);
var
 Index : array[0..4] of integer;
 i     : integer;
 n     : integer;
 Path  : string;
begin

  { comma index positions }
  n := 0;
  for i := 1 to Length(S) do
   if S[i] = ','
   then
     begin
       Index[n] := i;
       n := n + 1;
     end;

  { FileType 0 Path 1 Size 2 CreationTime 3 LastAccessTime 4  LastWriteTime}
  if Copy(S, 1, 1) = 'F' then FileType := 'File' else FileType := 'Folder';     { FileType }
  Path := Copy(S, 3, Index[1] - Index[0] - 1);                                  { Path }
  Name := ExtractFileName(Path);                                                { Name }
  Folder := ExtractFilePath(Path);                                              { Folder }
  SetLength(Folder, Length(Folder) - 1);
  Size           := Copy(S, Index[1] + 1, Index[2] - Index[1] - 1);             { Size }
  CreationTime   := Copy(S, Index[2] + 1, Index[3] - Index[2] - 1);             { CreationTime }
  LastAccessTime := Copy(S, Index[3] + 1, Index[4] - Index[3] - 1);             { LastAccessTime }
  LastWriteTime  := Copy(S, Index[4] + 1, Length(S) - Index[3] );               { LastWriteTime }

end;



{----------------------------------------------------------------------------------
 Description    : Searches the Folder and its subfolders if SubFoldersToo is True.
                  If finds any file having FileName and Extension then adds its
                  full path to the FileList (TStrings)
 Result         : The total size in bytes of founded files
 Parameters     :
                  Folder              : string, the starting folder, can not be empty
                  FileName            : string, it can be a name (MyFile) or empty or contain wildcards *, ?
                  Extension           : string, it can be an extension (txt) or empty or contain wildcards *, ?
                  SubFoldersToo       : boolean, if True then the routine search its subfolders
                  NewerThan           : integer, can be zero or greater, use FileAge Delphi function for that value
                  OlderThan           : integer, can be zero or greater, use FileAge Delphi function for that value
                  LargerThan          : Cardinal, can be zero or greater
                  SmallerThan         : Cardinal, can be zero or greater
                  var SearchedFiles   : Cardinal
                  var SearchedFolders : Cardinal
                  AbortMethod         : TAbortMethod, a method pointer for process interruption, can be nil
                  FileList            : TStrings, the result list having the below format:
                                        F or D to indicate file or directory found,
                                        the full path,
                                        the size,
                                        the creation datetime,
                                        the last access datetime,
                                        the last write datetime,
                                        and commas between them.
                                        The helper routine ParseFileInfoString can be use to parse this
                                        string to its individual parts

 Error checking : partially, be carefull
 Example call   : Get_FilesFromFolder('C:\Program Files\Borland\Delphi 3',
                                       '',
                                       'dcu',
                                       True,
                                       0,
                                       0,
                                       0,
                                       0,
                                       iSearchedFiles,
                                       iSearchedFolders,
                                       nil,
                                       ResultFileList,
                                       Memo1.Lines);
 Notes          :
                  1. if both NewerThan and OlderThan are greater to zero then NewerThan must be > OlderThan
                  2. if both LargerThan and SmallerThan are greater to zero then SmallerThan must be > LargerThan
-----------------------------------------------------------------------------------}
function Get_FilesFromFolder(const
                             Folder,
                             FileName,
                             Extension             : string;
                             SubFoldersToo         : boolean;
                             NewerThan             : integer;
                             OlderThan             : integer;
                             LargerThan            : Cardinal;
                             SmallerThan           : Cardinal;
                             var SearchedFiles     : Cardinal;
                             var SearchedFolders   : Cardinal;
                             AbortMethod           : TAbortMethod;
                             FileList              : TStrings) : integer;
type
  TFileType = (ftFile, ftFolder);
var
 SearchRec      : TSearchRec;
 FindResult     : Integer;
 Dir            : string;
 Name           : string;
 Ext            : string;
 AddIt          : boolean;
 S              : string;
 FileType       : TFileType;

   {----------------------------------------------------------------------------------
   Description    : converts a TFileTime ( Windows datetime format) to Delphi TDateTime
   Error checking : NO
   Notes          : TFileTime declaration from Wondows.pas
                    TFileTime = record
                      dwLowDateTime  : DWORD;
                      dwHighDateTime : DWORD;
                    end;
   -----------------------------------------------------------------------------------}
   function WinFileTimeToDelphiDateTime(FileTime:TFileTime):TDateTime;
   var
    LocalFileTime : TFileTime;
    DosFileTime   : DWORD;
   begin

     Result := 0;

     try
       { convert the file time based on the Coordinated Universal Time (UTC) to a local file time  }
        FileTimeToLocalFileTime(FileTime,                    { pointer to UTC file time to convert }
                             LocalFileTime);              { pointer to converted file time }


       { convert the 64-bit file time to MS-DOS date and time values }
        FileTimeToDosDateTime(LocalFileTime,                 { pointer to 64-bit file time }
                           LongRec(DosFileTime).Hi,       { pointer to 32-bit variable for MS-DOS date }
                           LongRec(DosFileTime).Lo);      { pointer to 32-bit variable for MS-DOS time }

       { convert the DOS date-time value to TDateTime format }
       Result := FileDateToDateTime(DosFileTime);
     except
       Abort;
     end;

   end;


begin
{$BOOLEVAL ON}
  Result := 0;

  Dir  := UpperCase(Folder);
  if Trim(Dir) = '' then raise Exception.Create('Folder can not be an empty string');
  if Dir[Length(Dir)] = '\' then SetLength(Dir, Length(Dir) - 1);

  Name := UpperCase(FileName);
  if ( Trim(Name) = '') or ( Trim(Name) = '*') then Name := '*';

  Ext := UpperCase(Extension);
  if ( Trim(Ext) = '' ) or ( Trim(Ext) = '*') then Ext := '*';

  S := Dir + '\' + Name + '.' + Ext;



  FindResult := FindFirst(S, faAnyFile, SearchRec);

  
  while FindResult = 0 do
  begin
    TRY

      { just to have a chance to abort the operation }
      Application.ProcessMessages;
      if Assigned(AbortMethod)
      then
        if AbortMethod
        then
          begin
            FileList.Clear;
            SysUtils.FindClose(SearchRec);
            Exit;
          end;


      { is a plain file or a folder? }
      if ((SearchRec.Name <> '.') and (SearchRec.Name <> '..'))
      then
        begin
          if ((SearchRec.Attr and faDirectory) > 0)
          then begin FileType := ftFolder; Inc(SearchedFolders); end
          else begin FileType := ftFile;   Inc(SearchedFiles);   end;
          AddIt := True;
        end;



      { add to list conditions }
      if AddIt then
      AddIt := (
                  ( ( OlderThan   = 0 ) and ( NewerThan   = 0 ) )
               or ( ( OlderThan   = 0 ) and ( NewerThan   > SearchRec.Time ))
               or ( ( NewerThan   = 0 ) and ( OlderThan   < SearchRec.Time ))
                   or (
                            (  ( NewerThan   > 0 ) and ( OlderThan   > 0) )
                       and  ( (OlderThan < SearchRec.Time ) and ( NewerThan > SearchRec.Time))

                       )
               )

          and (
                  ( ( SmallerThan   = 0 ) and ( LargerThan   = 0 ) )
               or ( ( SmallerThan   = 0 ) and ( LargerThan   < SearchRec.Size ))
               or ( ( LargerThan    = 0 ) and ( SmallerThan  > SearchRec.Size ))
                  or (
                            (  ( LargerThan   > 0 ) and ( SmallerThan   > 0) )
                       and ( (LargerThan < SearchRec.Size ) and ( SmallerThan > SearchRec.Size))

                           )
                );



      if AddIt
      then
        begin
          case FileType of
            ftFile    : begin
                          S := 'F,' + Dir + '\' + SearchRec.Name + ',' + IntToStr(SearchRec.Size)  + ',' +
                          DateTimeToStr(WinFileTimeToDelphiDateTime(SearchRec.FindData.ftCreationTime )) + ',' +
                          DateTimeToStr(WinFileTimeToDelphiDateTime(SearchRec.FindData.ftLastAccessTime )) + ',' +
                          DateTimeToStr(WinFileTimeToDelphiDateTime(SearchRec.FindData.ftLastWriteTime ));
                          Result := Result + SearchRec.Size;
                        end;
            ftFolder  : begin
                          S := 'D,' + Dir + '\' + SearchRec.Name + ',' + IntToStr(0)  + ',' +
                          DateTimeToStr(WinFileTimeToDelphiDateTime(SearchRec.FindData.ftCreationTime )) + ',' +
                          IntToStr(0) + ',' +
                          IntToStr(0);
                        end;
          end;
          FileList.Add( S );
          AddIt := False;
        end;
      Application.ProcessMessages;
    EXCEPT
     {$IFNDEF VER90}                             { Delphi 3-4 }
      on EOutOfResources do
        begin
          SysUtils.FindClose(SearchRec);
          ShowMessage('Operation failed');
          Abort;
        end
      else Result := Result + 0;
      {$ELSE}                                    { Delphi 2   }
         SysUtils.FindClose(SearchRec);
         ShowMessage('Operation failed');
         Abort;
      {$ENDIF}
    END;
    FindResult := FindNext(SearchRec);
  end;
  SysUtils.FindClose(SearchRec);


  if SubFoldersToo
  then
    begin
      FindResult := FindFirst(Dir +  '\*.*'  , faAnyFile, SearchRec);
      while FindResult = 0 do
      begin
        TRY
          if ((SearchRec.Name <> '.') and (SearchRec.Name <> '..'))
          then
            if ((SearchRec.Attr and faDirectory) > 0)
            then
              begin
                Result := Result + Get_FilesFromFolder(Dir + '\' + SearchRec.Name, FileName, Extension , SubFoldersToo,
                                                        NewerThan, OlderThan, LargerThan, SmallerThan,
                                                        SearchedFiles, SearchedFolders,
                                                        AbortMethod, FileList);
                Inc(SearchedFolders);
              end;
          Application.ProcessMessages;
        EXCEPT
          SysUtils.FindClose(SearchRec);
          ShowMessage('Operation failed');
          Abort;
        END;
        FindResult := FindNext(SearchRec);
      end;
      SysUtils.FindClose(SearchRec);
    end;


{$BOOLEVAL OFF}

end;






{===============================================================================
///                             TBTFileFinder                                 ///
===============================================================================}




{------------------------------------------------------------------------------}
constructor TBTFileFinder.Create;
begin
  inherited Create;
  FIsSearching           := False;

  FTotalFoundSize      := 0;
  FSearchMode          := smAll;        { smAll, smSelected, smCriteria }
  FSearchCriteria      := [];
  
  FIndividualFilesList := TStringList.Create;
  FFoldersList         := TStringList.Create;
  FChosenFilesList     := TStringList.Create;
  FChosenExtList       := TStringList.Create;
  
  FResultFileList            := TStringList.Create;
  FResultFileList.Sorted     := True;
  FResultFileList.Duplicates := dupIgnore;

end;

{------------------------------------------------------------------------------}
destructor TBTFileFinder.Destroy;
begin
  FResultFileList.Free;
  FChosenExtList.Free;
  FChosenFilesList.Free;
  FFoldersList.Free;
  FIndividualFilesList.Free;
  inherited Destroy;
end;


{------------------------------------------------------------------------------}
procedure TBTFileFinder.Stop;
begin
  FAbort := True;
end;


{------------------------------------------------------------------------------}
function  TBTFileFinder.StopSearching:boolean;
begin
  Result := ( FResultFileList.Count > 3000 ) or FAbort;
  if FResultFileList.Count > 3000
  then FResultFileList.Clear;
end;


{------------------------------------------------------------------------------}
procedure TBTFileFinder.Find_All;
var
 i : integer;
begin


  for i := 0 to FFoldersList.Count - 1 do
  begin
    if not DirectoryExists(FFoldersList[i])
    then RaiseFFErrorFmt(FOLDER_NOT_EXISTS_STR, [FFoldersList[i]])
    else
      FTotalFoundSize := FTotalFoundSize + Get_FilesFromFolder( FFoldersList[i],   { Folder         }
                                                                '*',               { FileName       }
                                                                '*',               { Extension      }
                                                                True,              { SubFoldersToo  }
                                                                0,                 { NewerThan      }
                                                                0,                 { OlderThan      }
                                                                0,                 { LargerThan     }
                                                                0,                 { SmallerThan    }
                                                                FSearchedFiles,    { iFiles         }
                                                                FSearchedFolders,  { iFolders       }
                                                                StopSearching,     { AbortMethod    }
                                                                FResultFileList    { List           }
                                                                );

  end;



end;


{------------------------------------------------------------------------------}
procedure TBTFileFinder.Find_Selected;
var
 i              : integer;
 sFolder,
 sFileName,
 sExtension      : string;
begin
                 

  for i := 0 to FIndividualFilesList.Count - 1 do
  begin
    sExtension := Copy(ExtractFileExt(FIndividualFilesList[i]), 2, Length(ExtractFileExt(FIndividualFilesList[i])) - 1);
    sFileName  := ChangeFileExt(ExtractFileName(FIndividualFilesList[i]), '');
    sFolder    := ExtractFilePath(FIndividualFilesList[i]);
    if sFolder[Length(sFolder)] = '\' then SetLength(sFolder, Length(sFolder) - 1);
    if sFolder = '' then sFolder := Copy(GetCurrentDir, 1, 3);
    FTotalFoundSize := FTotalFoundSize + Get_FilesFromFolder( sFolder,           { Folder         }
                                                              sFileName,         { FileName       }
                                                              sExtension,        { Extension      }
                                                              False,             { SubFoldersToo  }
                                                              0,                 { NewerThan      }
                                                              0,                 { OlderThan      }
                                                              0,                 { LargerThan     }
                                                              0,                 { SmallerThan    }
                                                              FSearchedFiles,    { iFiles         }
                                                              FSearchedFolders,  { iFolders       }
                                                              StopSearching,     { AbortMethod    }
                                                              FResultFileList    { List           }
                                                              );

  end;
end;

{------------------------------------------------------------------------------}
procedure TBTFileFinder.Find_Criteria;
type
 TFindMode = (fmFileName, fmExtension, fmNamesAndExt);
var
 iDir           : integer;
 iFile          : integer;
 iExt           : integer;

 sFolder         : string;
 sFileName       : string;
 sExtension      : string;

 bSubFoldersToo  : boolean;
 
 iNewerThan      : integer;
 iOlderThan      : integer;
 iLargerThan     : Cardinal;
 iSmallerThan    : Cardinal;
begin


  {  }
  bSubFoldersToo := scSubFoldersToo in FSearchCriteria;


  {  }
  if scNewerThan   in FSearchCriteria then iNewerThan   := FNewerThan   else iNewerThan   := 0;
  if scOlderThan   in FSearchCriteria then iOlderThan   := FOlderThan   else iOlderThan   := 0;
  if scLargerThan  in FSearchCriteria then iLargerThan  := FLargerThan  else iLargerThan  := 0;
  if scSmallerThan in FSearchCriteria then iSmallerThan := FSmallerThan else iSmallerThan := 0;

  {}
  if ( ( scOlderThan in FSearchCriteria ) and ( scNewerThan in FSearchCriteria ) )
  then
    if iNewerThan <  iOlderThan then RaiseFFError(OLDER_GREATER_THAN_NEWER_STR);
    

  {}
  if ( ( scSmallerThan in FSearchCriteria ) and ( scLargerThan in FSearchCriteria ) )
  then
    if iSmallerThan < iLargerThan then RaiseFFError(LARGER_GREATER_THAN_SMALLER_STR);

  {  }
  if FFoldersList.Count <= 0 then FFoldersList.Add(Copy(GetCurrentDir, 1, 2));

  if not ( scFileName in FSearchCriteria )
  then
    begin
      FChosenFilesList.Clear;
      FChosenFilesList.Add('*');
    end;

  if not ( scExtension in FSearchCriteria )
  then
    begin
      FChosenExtList.Clear;
      FChosenExtList.Add('*');
    end;      

  {  }
  for iDir := 0 to FFoldersList.Count - 1 do
    for iFile := 0 to FChosenFilesList.Count - 1 do
      for iExt := 0 to FChosenExtList.Count - 1 do
      begin
        sFolder    := FFoldersList[iDir];
        sFileName  := FChosenFilesList[iFile];
        sExtension := FChosenExtList[iExt];
        FTotalFoundSize := FTotalFoundSize + Get_FilesFromFolder( sFolder,
                                                                  sFileName,
                                                                  sExtension,
                                                                  bSubFoldersToo,
                                                                  iNewerThan,
                                                                  iOlderThan,
                                                                  iLargerThan,
                                                                  iSmallerThan,
                                                                  FSearchedFiles,
                                                                  FSearchedFolders,  
                                                                  StopSearching,     { AbortMethod    }
                                                                  FResultFileList );

      end;
 

end;


{------------------------------------------------------------------------------}
function TBTFileFinder.Execute:boolean;
begin

  Result := True;
  FSearchedFiles   := 0;
  FSearchedFolders := 0;
  FTotalFoundSize  := 0;
  FAbort           := False;


  try
    FIsSearching := True;
    case FSearchMode of
      smAll      : Find_All;
      smSelected : Find_Selected;
      smCriteria : if FSearchCriteria = [] then Find_All else Find_Criteria;
    end;
  except
    Result       := False;
    FIsSearching := False;
  end;
end;


{------------------------------------------------------------------------------}
procedure TBTFileFinder.ClearLists;
begin
  FResultFileList.Clear;
  FChosenExtList.Clear;
  FChosenFilesList.Clear;
  FFoldersList.Clear;
  FIndividualFilesList.Clear;
end;


{===============================================================================
                                       EOF
===============================================================================}
end.





































































